---
title: match / case 路由分支
---

# match / case 路由分支

`match / case` 适合“值 → 处理逻辑”的映射，默认 `hit_first`；也可以用 `hit_all` 做多标签路由。

## 场景：固定值路由

```python
from agently import TriggerFlow, TriggerFlowEventData

flow = TriggerFlow()

@flow.chunk
def input_level(_: TriggerFlowEventData):
  return "medium"

@flow.chunk
def low(_: TriggerFlowEventData):
  return "priority: low"

@flow.chunk
def medium(_: TriggerFlowEventData):
  return "priority: medium"

@flow.chunk
def high(_: TriggerFlowEventData):
  return "priority: high"

@flow.chunk
def unknown(_: TriggerFlowEventData):
  return "priority: unknown"

@flow.chunk
def print_result(data: TriggerFlowEventData):
  print(data.value)

(
  flow.to(input_level)
  .match()
  .case("low")
  .to(low)
  .case("medium")
  .to(medium)
  .case("high")
  .to(high)
  .case_else()
  .to(unknown)
  .end_match()
  .to(print_result)
)
flow.start(wait_for_result=False)
```

输出：

```text
priority: medium
```

## 场景：case 使用条件函数

```python
from agently import TriggerFlow, TriggerFlowEventData

flow = TriggerFlow()

@flow.chunk
def input_value(_: TriggerFlowEventData):
  return 3

@flow.chunk
def neg(_: TriggerFlowEventData):
  return "neg"

@flow.chunk
def small(_: TriggerFlowEventData):
  return "small"

@flow.chunk
def big(_: TriggerFlowEventData):
  return "big"

@flow.chunk
def print_result(data: TriggerFlowEventData):
  print(data.value)

(
  flow.to(input_value)
  .match()
  .case(lambda d: d.value < 0)
  .to(neg)
  .case(lambda d: d.value < 5)
  .to(small)
  .case_else()
  .to(big)
  .end_match()
  .to(print_result)
)
flow.start(wait_for_result=False)
```

输出：

```text
small
```

## 场景：mode="hit_all" 多标签路由

```python
from agently import TriggerFlow, TriggerFlowEventData

flow = TriggerFlow()

@flow.chunk
def input_value(_: TriggerFlowEventData):
  return 3

@flow.chunk
def small(_: TriggerFlowEventData):
  return "small"

@flow.chunk
def odd(_: TriggerFlowEventData):
  return "odd"

@flow.chunk
def print_result(data: TriggerFlowEventData):
  print(data.value)

(
  flow.to(input_value)
  .match(mode="hit_all")
  .case(lambda d: d.value < 5)
  .to(small)
  .case(lambda d: d.value % 2 == 1)
  .to(odd)
  .end_match()
  .to(print_result)
)
flow.start(wait_for_result=False)
```

输出：

```text
['small', 'odd']
```
