---
title: Context Engineering
---

# Context Engineering

“Context Engineering”强调的不是一句 Prompt，而是**整段上下文的设计与治理**。英文资料中对这个概念的共识包括：

- **Context 是推理时进入模型的全部 tokens**，并不只指 system prompt 或指令本身。
- Context Engineering 是对这些 tokens 的**筛选、组织、更新与迭代**，目的是在有限上下文内获得稳定行为。
- 典型构成包含：系统指令、用户输入、工具/函数描述、外部数据、历史状态、结构化输入输出等。

> 参考资料：Anthropic “Effective context engineering for AI agents”；Prompting Guide “Context Engineering Guide”。

下面结合这些观点，客观说明 Agently 在 Context Engineering 上的支持与实践方式，并给出可运行的代码示例。

## 1) 分层上下文：Agent 级别与 Request 级别

将“稳定背景”与“本次请求”拆开，有助于控制上下文噪音与复用成本。

```python
from agently import Agently

agent = Agently.create_agent()

# Agent 级别：长期稳定的上下文
agent.set_agent_prompt("system", "你是企业知识助手")
agent.set_agent_prompt("instruct", ["回答必须简短且可操作"]) 

# Request 级别：本次临时输入
result = (
  agent
  .set_request_prompt("input", "给我一句部署建议")
  .output({"advice": ("str", "一句话建议")})
  .start()
)

print(result)
```

## 2) 动态变量与模板化上下文

Context Engineering 要求“动态信息”可结构化地注入。

```python
from agently import Agently

agent = Agently.create_agent()

user = {"name": "Moxin", "role": "PM"}
agent.set_request_prompt(
  "input",
  "为{name}（{role}）总结本次会议决策",
  mappings=user,
)
agent.set_request_prompt("output", {"summary": ("str", "一句话总结")})

print(agent.start())
```

## 3) 配置化 Prompt：上下文可维护

将上下文从代码中拆分出来，便于版本化与运营协作。

```yaml
# prompt.yaml
.agent:
  system: "你是企业知识助手"
  instruct:
    - "先给结论，再给依据"
.request:
  input: "{question}"
  output:
    summary:
      $type: str
      $desc: "一句话结论"
```

```python
from agently import Agently

agent = Agently.create_agent()
agent.load_yaml_prompt("prompt.yaml", mappings={"question": "Agently 有哪些优势？"})

print(agent.start())
```

## 4) 结构化输入/输出：降低歧义与噪音

结构化输出是 Context Engineering 中“结构化约束”的重要组成部分，可让结果更稳定、更易复用。

```python
from agently import Agently

agent = Agently.create_agent()

response = (
  agent
  .set_request_prompt("input", "写一个发布计划，包含目标和里程碑")
  .output({
    "goal": ("str", "目标"),
    "milestones": [
      {"title": ("str", "里程碑"), "date": ("str", "日期")}
    ]
  })
  .get_response()
)

print(response.get_data(ensure_keys=["goal", "milestones"]))
```

## 5) 外部知识注入（RAG 入口）

Context Engineering 的核心之一是“把对的知识带进上下文”。Agently 支持以结构化方式注入外部检索结果。

```python
from agently import Agently

agent = Agently.create_agent()

# 示例：本地检索
kb = [
  {"title": "Agently v4", "content": "支持 Output Format 与 TriggerFlow"},
  {"title": "Agently Prompt", "content": "支持 YAML/JSON 配置化"},
]

def retrieve(query: str):
  return [item for item in kb if "Agently" in item["title"]]

question = "Agently 的核心能力是什么？"
knowledge = retrieve(question)

result = (
  agent
  .set_request_prompt("input", question)
  .set_request_prompt("info", {"knowledge": knowledge})
  .output({"answer": ("str", "回答"), "sources": ["str"]})
  .start()
)

print(result)
```

## 6) 记忆与历史：控制上下文长度

多轮场景的上下文会持续膨胀，Agently 的 Session/Memo 支持“历史压缩与记忆提炼”。

```python
from agently import Agently
from agently.core import Session

agent = Agently.create_agent()
session = Session(agent=agent)

session.append_message({"role": "user", "content": "我只要简短回答"})
session.append_message({"role": "assistant", "content": "明白"})

# 通过 resize 控制上下文大小，并同步 memo
session.set_settings("session.resize.max_messages_text_length", 800)
session.set_settings("session.memo.enabled", True)

session.resize()
agent.set_chat_history(session.current_chat_history)

print(agent.input("复述我的偏好").start())
```

## 7) 工具与函数：将能力作为上下文组件

工具描述也是上下文的一部分。Agently 支持将工具规范化注入模型上下文，并按需调用。

```python
from agently import Agently

agent = Agently.create_agent()

def fetch_order(order_id: str) -> dict:
  return {"order_id": order_id, "status": "paid"}

agent.register_tool(
  name="fetch_order",
  desc="查询订单状态",
  kwargs={"order_id": (str, "订单号")},
  func=fetch_order,
  returns=dict,
)
agent.use_tools("fetch_order")

print(
  agent
  .input("查询订单 A-100 的状态")
  .output({"status": ("str", "状态")})
  .start()
)
```

## 8) 评估与迭代（可集成外部评估）

Context Engineering 往往需要迭代验证。Agently 没有内置评估平台，但结果对象支持接入自定义评估。

```python
from agently import Agently

agent = Agently.create_agent()
response = agent.input("用一句话解释可观测性").get_response()

text = response.get_text()
score = 1 if len(text) <= 30 else 0  # 简单长度约束示例

print(text)
print("score=", score)
```

## 参考英文资料

- https://www.anthropic.com/engineering/effective-context-engineering-for-ai-agents
- https://www.promptingguide.ai/guides/context-engineering-guide
