---
title: 问卷调查多轮对话
---

# 问卷调查多轮对话

## 场景

需要在多轮对话里收集目标、场景与风险点，并最终输出一段可复用的访谈总结。

## 方案

- TriggerFlow 用 `for_each` 串行组织问答  
- Agent 负责“问法”优化与最终总结  
- 所有问答结果作为结构化数据回传  

```mermaid
flowchart LR
  Q[问题列表] --> A1[生成问法]
  A1 --> A2[记录回答]
  A2 --> S[汇总总结]
```

## 代码

```python
from agently import Agently, TriggerFlow, TriggerFlowEventData

Agently.set_settings("prompt.add_current_time", False)
Agently.set_settings("OpenAICompatible", {
  "base_url": "http://localhost:11434/v1",
  "model": "qwen2.5:7b",
  "model_type": "chat",
})

agent = Agently.create_agent()

questions = [
  {
    "id": "q1",
    "intent": "了解目标",
    "question": "你希望这个工具帮你解决什么问题？",
    "answer": "快速把 PRD 变成测试用例，并能导出到 Jira。",
  },
  {
    "id": "q2",
    "intent": "了解场景",
    "question": "通常在哪些环节会使用这个工具？",
    "answer": "需求评审后，测试排期前。",
  },
  {
    "id": "q3",
    "intent": "了解限制",
    "question": "最不能接受的输出问题是什么？",
    "answer": "测试步骤不完整或缺少边界条件。",
  },
]

flow = TriggerFlow()

@flow.chunk
def input_questions(_: TriggerFlowEventData):
  return questions

@flow.chunk
async def ask_and_collect(data: TriggerFlowEventData):
  item = data.value
  ask = await (
    agent
    .input({"intent": item["intent"], "question": item["question"]})
    .instruct("你是问卷调查助手，用更自然的方式提问，保持简短。")
    .output({"ask": ("str", "问法"), "tone": ("str", "语气关键词")})
    .async_start()
  )
  return {"ask": ask["ask"], "answer": item["answer"], "intent": item["intent"]}

@flow.chunk
async def summarize(data: TriggerFlowEventData):
  qa_list = data.value
  result = await (
    agent
    .input({"qa": qa_list})
    .instruct("基于问答总结用户目标、场景与风险点，输出 3 条要点。")
    .output({"summary": [("str", "要点")]})
    .async_start()
  )
  return {"qa": qa_list, "summary": result["summary"]}

(
  flow.to(input_questions)
  .for_each(concurrency=1)
  .to(ask_and_collect)
  .end_for_each()
  .to(summarize)
  .end()
)

result = flow.start("run")
print(result)
```

## 运行结果

```text
{'qa': [{'ask': '您希望通过这个工具达成什么目标呢？', 'answer': '快速把 PRD 变成测试用例，并能导出到 Jira。', 'intent': '了解目标'}, {'ask': '这个工具通常会在哪些环节用到呢？', 'answer': '需求评审后，测试排期前。', 'intent': '了解场景'}, {'ask': '哪些问题是你们最不能接受的呢？', 'answer': '测试步骤不完整或缺少边界条件。', 'intent': '了解限制'}], 'summary': ['尽快将PRD转化为完备的测试用例，并能够导出到Jira系统中。', '在需求评审之后、测试计划之前使用该工具。', '确保测试步骤完整且包含边界条件，避免遗漏关键边界情况。']}
```
